<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Function to send JSON response
function sendJsonResponse($success, $message, $data = null) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ]);
    exit;
}

// Check if this is an AJAX request
$isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';

// Check if user has any customer-related permissions
$customerPermissions = [
    'view_customers', 'manage_customers', 'add_customer', 'edit_customer', 'delete_customer',
    'manage_credit', 'process_credit_payment', 'waive_credit', 'send_reminder', 'add_credit_item'
];

if (!hasAnyPermission($customerPermissions)) {
    if ($isAjax) {
        sendJsonResponse(false, "You don't have permission to access customer management.");
    } else {
        $_SESSION['error'] = "You don't have permission to access customer management.";
        header("Location: dashboard.php");
        exit;
    }
}

// Set error handler for AJAX requests
if ($isAjax) {
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        sendJsonResponse(false, "PHP Error: " . $errstr);
    });
}

$activePage = 'customers';

$success = '';
$error = '';
$search = $_GET['search'] ?? '';
$sort = $_GET['sort'] ?? 'name';
$order = $_GET['order'] ?? 'asc';
$category = $_GET['category'] ?? 'all';

// Get credit system settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'credit_settings'");
$stmt->execute();
$creditSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$creditSystemEnabled = isset($creditSettings['enable_credit_system']) && $creditSettings['enable_credit_system'] == '1';
$defaultCreditLimit = $creditSettings['default_credit_limit'] ?? '0';
$maxCreditLimit = $creditSettings['max_credit_limit'] ?? '10000';

// Handle customer actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_customer'])) {
        // Check for duplicate name, phone, or email
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM customers WHERE name = ? OR phone = ? OR email = ?");
        $stmt->execute([
            trim($_POST['name']),
            trim($_POST['phone']),
            trim($_POST['email'])
        ]);
        $exists = $stmt->fetchColumn();
        if ($exists > 0) {
            $error = "Customer name, phone number, or email already exists in the database.";
            if ($isAjax) {
                sendJsonResponse(false, $error);
            }
        } else {
            try {
                $creditLimit = $creditSystemEnabled ? min(floatval($_POST['credit_limit']), floatval($maxCreditLimit)) : 0;
                $stmt = $pdo->prepare("INSERT INTO customers (name, phone, email, address, credit_limit, status) VALUES (?, ?, ?, ?, ?, 'active')");
                $stmt->execute([
                    trim($_POST['name']),
                    trim($_POST['phone']),
                    trim($_POST['email']),
                    trim($_POST['address']),
                    $creditLimit
                ]);
                $customerId = $pdo->lastInsertId();
                $success = "Customer added successfully!";
                if ($isAjax) {
                    sendJsonResponse(true, $success, ['id' => $customerId]);
                }
            } catch (PDOException $e) {
                $error = "Error adding customer: " . $e->getMessage();
                if ($isAjax) {
                    sendJsonResponse(false, $error);
                }
            }
        }
    } elseif (isset($_POST['edit_customer'])) {
        try {
            // Validate required fields
            if (empty($_POST['name']) || empty($_POST['phone'])) {
                throw new Exception("Name and phone are required fields.");
            }

            // Check for duplicate name, phone, or email (excluding current customer)
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM customers WHERE (name = ? OR phone = ? OR email = ?) AND id != ?");
            $stmt->execute([
                trim($_POST['name']),
                trim($_POST['phone']),
                trim($_POST['email']),
                intval($_POST['id'])
            ]);
            $exists = $stmt->fetchColumn();
            if ($exists > 0) {
                throw new Exception("Customer name, phone number, or email already exists in the database.");
            }

            // Get current credit limit and credit usage
            $stmt = $pdo->prepare("
                SELECT c.credit_limit, 
                       COALESCE(SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END), 0) as total_credit_used,
                       COALESCE(SUM(cp.amount), 0) as total_paid_and_waived,
                       (
                         SELECT 
                           COALESCE(SUM(ct2.amount) - COALESCE((SELECT SUM(cp2.amount) FROM credit_payments cp2 WHERE cp2.credit_transaction_id = ct2.id), 0), 0)
                         FROM credit_transactions ct2
                         WHERE ct2.customer_id = c.id
                       ) as total_balance,
                       MAX(ct.created_at) as last_credit_activity
                FROM customers c
                LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
                LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
                WHERE c.id = ?
                GROUP BY c.id
            ");
            $stmt->execute([intval($_POST['id'])]);
            $currentCredit = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$currentCredit) {
                throw new Exception("Customer not found");
            }

            // Only update credit limit if it's enabled and within max limit
            $creditLimit = $currentCredit['credit_limit']; // Maintain current limit by default
            if ($creditSystemEnabled) {
                $newCreditLimit = floatval($_POST['credit_limit']);
                if ($newCreditLimit <= floatval($maxCreditLimit)) {
                    $creditLimit = $newCreditLimit;
                } else {
                    throw new Exception("Credit limit cannot exceed the maximum limit of " . $currency . " " . number_format($maxCreditLimit, 2));
                }
            }

            // Update customer details
            $stmt = $pdo->prepare("UPDATE customers SET name = ?, phone = ?, email = ?, address = ?, credit_limit = ? WHERE id = ?");
            $stmt->execute([
                trim($_POST['name']),
                trim($_POST['phone']),
                trim($_POST['email']),
                trim($_POST['address']),
                $creditLimit,
                intval($_POST['id'])
            ]);

            // Calculate new credit status
            $totalCreditUsed = floatval($currentCredit['total_credit_used']);
            $totalPaidAndWaived = floatval($currentCredit['total_paid_and_waived']);
            $availableCredit = $creditLimit - ($totalCreditUsed - $totalPaidAndWaived);
            $creditUsagePercent = $creditLimit > 0 ? (($totalCreditUsed - $totalPaidAndWaived) / $creditLimit) * 100 : 0;
            
            // Determine credit status
            if ($creditLimit > 0) {
                if ($creditUsagePercent >= 90) {
                    $creditStatus = 'critical';
                    $statusText = 'Critical';
                } elseif ($creditUsagePercent >= 75) {
                    $creditStatus = 'warning';
                    $statusText = 'Warning';
                } elseif ($creditUsagePercent >= 50) {
                    $creditStatus = 'moderate';
                    $statusText = 'Moderate';
                } else {
                    $creditStatus = 'good';
                    $statusText = 'Good';
                }
            } else {
                $creditStatus = 'none';
                $statusText = 'No Credit';
            }

            $success = "Customer updated successfully!";
            
            // Return updated data for AJAX refresh
            if ($isAjax) {
                sendJsonResponse(true, $success, [
                    'credit_status' => $creditStatus,
                    'status_text' => $statusText,
                    'credit_usage_percent' => $creditUsagePercent,
                    'available_credit' => $availableCredit,
                    'last_credit_activity' => $currentCredit['last_credit_activity']
                ]);
            }
        } catch (Exception $e) {
            $error = "Error updating customer: " . $e->getMessage();
            if ($isAjax) {
                sendJsonResponse(false, $error);
            }
        }
    } elseif (isset($_POST['delete_customer'])) {
        try {
            // First check if customer has any associated sales or credit transactions
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM sales WHERE customer_id = ?");
            $stmt->execute([intval($_POST['id'])]);
            $hasSales = $stmt->fetchColumn() > 0;

            $stmt = $pdo->prepare("SELECT COUNT(*) FROM credit_transactions WHERE customer_id = ?");
            $stmt->execute([intval($_POST['id'])]);
            $hasCredit = $stmt->fetchColumn() > 0;

            if ($hasSales || $hasCredit) {
                $error = "Cannot delete customer with existing sales or credit transactions.";
            } else {
                $stmt = $pdo->prepare("DELETE FROM customers WHERE id = ?");
                $stmt->execute([intval($_POST['id'])]);
                $success = "Customer deleted successfully!";
            }
        } catch (PDOException $e) {
            $error = "Error deleting customer: " . $e->getMessage();
        }
    } elseif (isset($_POST['suspend_customer'])) {
        try {
            $stmt = $pdo->prepare("UPDATE customers SET status = 'suspended' WHERE id = ?");
            $stmt->execute([intval($_POST['suspend_customer'])]);
            $success = "Customer suspended successfully!";
        } catch (PDOException $e) {
            $error = "Error suspending customer: " . $e->getMessage();
        }
    } elseif (isset($_POST['reactivate_customer'])) {
        try {
            $stmt = $pdo->prepare("UPDATE customers SET status = 'active' WHERE id = ?");
            $stmt->execute([intval($_POST['reactivate_customer'])]);
            $success = "Customer reactivated successfully!";
        } catch (PDOException $e) {
            $error = "Error reactivating customer: " . $e->getMessage();
        }
    }
}

// Build query for customers with credit information
$query = "SELECT c.*, 
           COALESCE(credit_summary.total_credit_used, 0) as total_credit_used,
           COALESCE(credit_summary.total_paid_and_waived, 0) as total_paid_and_waived,
           COALESCE(credit_summary.total_balance, 0) as total_balance,
           (
             -- Cash sales (only immediate cash payments, not credit sales that were later paid)
             SELECT COALESCE(SUM(sp.amount), 0)
             FROM sales s1
             JOIN sale_payments sp ON s1.id = sp.sale_id
             WHERE s1.customer_id = c.id 
               AND sp.method = 'cash'
           )
           +
           (
             -- Actual payments made on credit sales (not waived)
             SELECT COALESCE(SUM(cp.amount), 0)
             FROM credit_transactions ct2
             JOIN credit_payments cp ON cp.credit_transaction_id = ct2.id
             WHERE ct2.customer_id = c.id
               AND cp.is_waived = 0
           ) as total_spent_final,
          (
             SELECT COUNT(DISTINCT s_sub.id)
             FROM sales s_sub
             WHERE s_sub.customer_id = c.id
           ) as total_sales,
           (
             SELECT COALESCE(SUM(s_sub.total_amount), 0)
             FROM sales s_sub
             WHERE s_sub.customer_id = c.id
           ) as total_spent,
           (
             SELECT MAX(ct_sub.created_at)
             FROM credit_transactions ct_sub
             WHERE ct_sub.customer_id = c.id
           ) as last_credit_activity
    FROM customers c
    LEFT JOIN (
        SELECT 
            ct.customer_id,
            SUM(ct.amount) as total_credit_used,
            SUM(COALESCE(paid_amounts.paid, 0) + COALESCE(waived_amounts.waived, 0)) as total_paid_and_waived,
            SUM(ct.amount - COALESCE(paid_amounts.paid, 0) - COALESCE(waived_amounts.waived, 0)) as total_balance
        FROM credit_transactions ct
        LEFT JOIN (
            SELECT 
                cp.credit_transaction_id,
                SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END) as paid
            FROM credit_payments cp
            GROUP BY cp.credit_transaction_id
        ) paid_amounts ON ct.id = paid_amounts.credit_transaction_id
        LEFT JOIN (
            SELECT 
                cp.credit_transaction_id,
                SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END) as waived
            FROM credit_payments cp
            GROUP BY cp.credit_transaction_id
        ) waived_amounts ON ct.id = waived_amounts.credit_transaction_id
        WHERE ct.type = 'credit'
        AND (ct.amount - COALESCE(paid_amounts.paid, 0) - COALESCE(waived_amounts.waived, 0)) > 0.001
        GROUP BY ct.customer_id
    ) credit_summary ON c.id = credit_summary.customer_id
          WHERE 1=1";

$params = [];

if ($search) {
    $query .= " AND (c.name LIKE ? OR c.phone LIKE ? OR c.email LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($category && $category !== 'all') {
    if ($category === 'active') {
        $query .= " AND c.status = 'active'";
    } elseif ($category === 'suspended') {
        $query .= " AND c.status = 'suspended'";
    } elseif ($category === 'with_credit') {
        $query .= " AND c.credit_limit > 0";
    } elseif ($category === 'no_credit') {
        $query .= " AND c.credit_limit = 0";
    }
}

// Add sorting
$query .= " ORDER BY " . ($sort === 'name' ? 'c.name' : 
                        ($sort === 'orders' ? 'total_sales' : 
                        ($sort === 'spent' ? 'total_spent' : 'c.name'))) . 
         " " . ($order === 'desc' ? 'DESC' : 'ASC');

// Get customers
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$customers = $stmt->fetchAll();

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Calculate statistics
$total_customers = count($customers);
$total_orders = 0;
$total_revenue = 0;
$total_credit = 0;

foreach ($customers as $customer) {
    $total_orders += $customer['total_sales'];
    $total_revenue += $customer['total_spent'];
    // Use max credit limit instead of individual customer limits
    $total_credit = floatval($maxCreditLimit);
}

// Calculate credit status for each customer
foreach ($customers as &$customer) {
    $customer['total_credit_used'] = floatval($customer['total_credit_used']);
    $customer['total_paid_and_waived'] = floatval($customer['total_paid_and_waived']);
    $customer['credit_limit'] = floatval($customer['credit_limit']);
    $customer['total_balance'] = floatval($customer['total_balance']);

    // Use total_balance for percentage and status
    $customer['credit_usage_percent'] = $customer['credit_limit'] > 0 ?
        ($customer['total_balance'] / $customer['credit_limit']) * 100 : 0;

    if ($customer['credit_limit'] > 0) {
        if ($customer['credit_usage_percent'] >= 90) {
            $customer['credit_status'] = 'critical';
            $customer['status_text'] = 'Critical';
        } elseif ($customer['credit_usage_percent'] >= 75) {
            $customer['credit_status'] = 'warning';
            $customer['status_text'] = 'Warning';
        } elseif ($customer['credit_usage_percent'] >= 50) {
            $customer['credit_status'] = 'moderate';
            $customer['status_text'] = 'Moderate';
        } elseif ($customer['credit_usage_percent'] > 0) {
            $customer['credit_status'] = 'good';
            $customer['status_text'] = 'Good';
        } else {
            $customer['credit_status'] = 'good';
            $customer['status_text'] = 'Good';
        }
    } else {
        $customer['credit_status'] = 'none';
        $customer['status_text'] = 'No Credit';
    }
}
unset($customer);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customers - EDDUH COMPUTERS</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { 
            background: #f8fafc; 
            font-family: 'Inter', sans-serif; 
        }
        .card { 
            border-radius: 16px; 
            box-shadow: 0 4px 20px rgba(0,0,0,0.05); 
            border: none; 
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .card-header { 
            background: none; 
            border-bottom: none; 
            padding: 1.5rem 1.5rem 0.5rem;
        }
        .card-body {
            padding: 1.5rem;
        }
        .stat-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-left: 4px solid;
        }
        .stat-card.primary { border-left-color: #3b82f6; }
        .stat-card.success { border-left-color: #10b981; }
        .stat-card.info { border-left-color: #06b6d4; }
        .stat-card.warning { border-left-color: #f59e0b; }
        .stat-value {
            font-size: 1.8rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0.5rem 0;
        }
        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        .table {
            background: #fff;
            border-radius: 12px;
            overflow: hidden;
        }
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #475569;
            border-bottom: 2px solid #e2e8f0;
        }
        .table td {
            vertical-align: middle;
            color: #334155;
        }
        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }
        .btn-primary {
            background: #3b82f6;
            border: none;
        }
        .btn-primary:hover {
            background: #2563eb;
        }
        .filter-card {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        .filter-card .form-control {
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        .filter-card .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .badge {
            padding: 0.5em 1em;
            border-radius: 6px;
            font-weight: 500;
        }
        .badge-credit {
            background: #10b981;
            color: white;
        }
        .badge-no-credit {
            background: #64748b;
            color: white;
        }
        .progress {
            height: 8px;
            border-radius: 4px;
            background: #e2e8f0;
        }
        .progress-bar {
            border-radius: 4px;
        }
        .modal-content {
            border-radius: 16px;
            border: none;
        }
        .modal-header {
            border-bottom: none;
            padding: 1.5rem;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .modal-footer {
            border-top: none;
            padding: 1.5rem;
        }
        .customer-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e2e8f0;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            color: #64748b;
        }
        #successModal .modal-content {
            border-radius: 15px;
            overflow: hidden;
        }
        
        #successModal .modal-header {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        }
        
        #successModal .modal-body {
            font-size: 1.1rem;
            font-weight: 500;
            color: #495057;
        }
        
        #successModal .progress {
            border-radius: 0;
            background-color: #e9ecef;
        }
        
        #successModal .progress-bar {
            transition: width 3s linear;
        }
        
        #successModal .btn-close-white {
            filter: brightness(0) invert(1);
        }
    </style>
</head>
<body>
    <!-- Edit Customer Modal -->
    <div class="modal fade" id="editCustomerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Customer</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form method="post" id="editCustomerForm">
                        <input type="hidden" name="id" id="edit_id">
                        <div class="mb-3">
                            <label class="form-label">Name</label>
                            <input type="text" class="form-control" name="name" id="edit_name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Phone</label>
                            <input type="tel" class="form-control" name="phone" id="edit_phone" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" id="edit_email">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Address</label>
                            <textarea class="form-control" name="address" id="edit_address" rows="2"></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Credit Limit</label>
                            <div class="input-group">
                                <span class="input-group-text"><?php echo $currency; ?></span>
                                <input type="number" class="form-control" name="credit_limit" id="edit_credit_limit"
                                       min="0" max="<?php echo $maxCreditLimit; ?>" step="0.01" <?php if (!$creditSystemEnabled) echo 'readonly'; ?>>
                            </div>
                            <small class="text-muted">Maximum credit limit is <?php echo $currency . ' ' . number_format($maxCreditLimit, 2); ?></small>
                        </div>
                        <div class="text-end">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="edit_customer" class="btn btn-primary" data-permission="edit_customer" data-permission-label="Edit Customer">Save Changes</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                <div class="container-fluid mt-4">
                    <!-- Page Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2 class="mb-0">Customers</h2>
                        <div>
                            <a href="suppliers.php" class="btn btn-info me-2">
                                <i class="fas fa-truck"></i> Supplies
                            </a>
                            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCustomerModal" data-permission="add_customer" data-permission-label="Add Customer">
                                <i class="fas fa-plus"></i> Add Customer
                            </button>
                        </div>
                    </div>

                    <!-- Statistics Cards -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <div class="card stat-card primary">
                                <div class="card-body">
                                    <div class="stat-label">Total Customers</div>
                                    <div class="stat-value"><?php echo number_format($total_customers); ?></div>
                                    <div class="progress mt-2">
                                        <div class="progress-bar bg-primary" style="width: 100%"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stat-card success">
                                <div class="card-body">
                                    <div class="stat-label">Total Orders</div>
                                    <div class="stat-value"><?php echo number_format($total_orders); ?></div>
                                    <div class="progress mt-2">
                                        <div class="progress-bar bg-success" style="width: 100%"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stat-card info">
                                <div class="card-body">
                                    <div class="stat-label">Total Revenue</div>
                                    <div class="stat-value"><?php echo $currency . ' ' . number_format($total_revenue, 2); ?></div>
                                    <div class="progress mt-2">
                                        <div class="progress-bar bg-info" style="width: 100%"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="card stat-card warning">
                                <div class="card-body">
                                    <div class="stat-label">Maximum Credit Limit</div>
                                    <div class="stat-value"><?php echo $currency . ' ' . number_format($maxCreditLimit, 2); ?></div>
                                    <div class="progress mt-2">
                                        <div class="progress-bar bg-warning" style="width: 100%"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Filters -->
                    <div class="card mb-4">
                        <div class="card-body">
                            <form method="GET" class="row g-3" id="filterForm">
                                <div class="col-md-4">
                                    <label class="form-label">Search</label>
                                    <input type="text" name="search" class="form-control" placeholder="Search by name, phone, or email..." value="<?php echo htmlspecialchars($search); ?>" id="searchInput">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">Category</label>
                                    <select name="category" class="form-select" id="categorySelect">
                                        <option value="all" <?php echo $category === 'all' ? 'selected' : ''; ?>>All Customers</option>
                                        <option value="active" <?php echo $category === 'active' ? 'selected' : ''; ?>>Active Customers</option>
                                        <option value="suspended" <?php echo $category === 'suspended' ? 'selected' : ''; ?>>Suspended Customers</option>
                                        <option value="with_credit" <?php echo $category === 'with_credit' ? 'selected' : ''; ?>>With Credit</option>
                                        <option value="no_credit" <?php echo $category === 'no_credit' ? 'selected' : ''; ?>>No Credit</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">Sort By</label>
                                    <select name="sort" class="form-select" id="sortSelect">
                                        <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Name</option>
                                        <option value="orders" <?php echo $sort === 'orders' ? 'selected' : ''; ?>>Total Orders</option>
                                        <option value="spent" <?php echo $sort === 'spent' ? 'selected' : ''; ?>>Total Spent</option>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Order</label>
                                    <select name="order" class="form-select" id="orderSelect">
                                        <option value="asc" <?php echo $order === 'asc' ? 'selected' : ''; ?>>Ascending</option>
                                        <option value="desc" <?php echo $order === 'desc' ? 'selected' : ''; ?>>Descending</option>
                                    </select>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Customers Table -->
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="card-title mb-0">Customers List</h5>
                            <span class="badge bg-primary"><?php echo count($customers); ?> Customers</span>
                        </div>
                        <div class="card-body p-0">
                        <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                            <th>Customer</th>
                                        <th>Contact</th>
                                            <th>Orders</th>
                                            <th>Total Spent</th>
                                        <th>Credit Status</th>
                                        <th>Credit Amount Used</th>
                                        <th>Last Activity</th>
                                            <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($customers as $customer): ?>
                                        <tr data-customer-id="<?php echo $customer['id']; ?>">
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="customer-avatar me-2">
                                                        <?php echo strtoupper(substr($customer['name'], 0, 1)); ?>
                                                    </div>
                                                    <div>
                                                        <div class="fw-bold"><?php echo htmlspecialchars($customer['name']); ?></div>
                                                        <small class="text-muted"><?php echo htmlspecialchars($customer['address']); ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <div><?php echo htmlspecialchars($customer['phone']); ?></div>
                                                <small class="text-muted"><?php echo htmlspecialchars($customer['email']); ?></small>
                                            </td>
                                            <td><?php echo number_format($customer['total_sales']); ?></td>
                                            <td><?php echo $currency . ' ' . number_format($customer['total_spent_final'], 2); ?></td>
                                            <td class="credit-status">
                                                <?php if ($customer['credit_limit'] > 0): ?>
                                                    <div class="d-flex align-items-center">
                                                        <div class="progress flex-grow-1 me-2" style="height: 6px;">
                                                            <div class="progress-bar <?php 
                                                                echo $customer['credit_status'] === 'critical' ? 'bg-danger' : 
                                                                    ($customer['credit_status'] === 'warning' ? 'bg-warning' : 
                                                                    ($customer['credit_status'] === 'moderate' ? 'bg-info' : 'bg-success')); 
                                                            ?>" style="width: <?php echo $customer['credit_usage_percent']; ?>%"></div>
                                                        </div>
                                                        <span class="badge <?php 
                                                            echo $customer['credit_status'] === 'critical' ? 'bg-danger' : 
                                                                ($customer['credit_status'] === 'warning' ? 'bg-warning' : 
                                                                ($customer['credit_status'] === 'moderate' ? 'bg-info' : 'bg-success')); 
                                                        ?>">
                                                            <?php echo $customer['status_text']; ?>
                                                        </span>
                                                    </div>
                                                    <small class="text-muted">
                                                        <?php echo number_format($customer['credit_usage_percent'], 1); ?>% used
                                                    </small>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">No Credit</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="available-credit">
                                                <?php if ($customer['credit_limit'] > 0): ?>
                                                    <div class="<?php echo $customer['total_balance'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                                        <?php echo $currency . ' ' . number_format($customer['total_balance'], 2); ?>
                                                    </div>
                                                    <small class="text-muted">
                                                        of <?php echo $currency . ' ' . number_format($customer['credit_limit'], 2); ?>
                                                    </small>
                                                <?php else: ?>
                                                    -
                                                <?php endif; ?>
                                            </td>
                                            <td class="last-activity">
                                                <?php if ($customer['last_credit_activity']): ?>
                                                    <div><?php echo date('M d, Y', strtotime($customer['last_credit_activity'])); ?></div>
                                                    <small class="text-muted">
                                                        <?php 
                                                            $lastActivity = strtotime($customer['last_credit_activity']);
                                                            $now = time();
                                                            $diff = $now - $lastActivity;
                                                            $days = floor($diff / (60 * 60 * 24));
                                                            echo $days . ' days ago';
                                                        ?>
                                                    </small>
                                                <?php else: ?>
                                                    -
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                <button type="button" class="btn btn-sm btn-outline-primary edit-customer" 
                                                        data-id="<?php echo $customer['id']; ?>"
                                                        data-name="<?php echo htmlspecialchars($customer['name']); ?>"
                                                        data-phone="<?php echo htmlspecialchars($customer['phone']); ?>"
                                                        data-email="<?php echo htmlspecialchars($customer['email']); ?>"
                                                        data-address="<?php echo htmlspecialchars($customer['address']); ?>"
                                                        data-credit="<?php echo $customer['credit_limit']; ?>"
                                                        data-permission="edit_customer" data-permission-label="Edit Customer">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <a href="print_customer_sales.php?id=<?php echo $customer['id']; ?>" 
                                                       class="btn btn-sm btn-outline-info" 
                                                       target="_blank"
                                                       data-permission="view_customers" 
                                                       data-permission-label="View Customers">
                                                        <i class="fas fa-print"></i>
                                                    </a>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this customer?');">
                                                    <input type="hidden" name="id" value="<?php echo $customer['id']; ?>">
                                                    <button type="submit" name="delete_customer" class="btn btn-sm btn-outline-danger" data-permission="delete_customer" data-permission-label="Delete Customer">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Customer Modal -->
    <div class="modal fade" id="addCustomerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                        <h5 class="modal-title">Add New Customer</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                    <div class="modal-body">
                    <form method="post" id="addCustomerForm" class="needs-validation" novalidate>
                        <div class="mb-3">
                            <label class="form-label">Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="name" required>
                            <div class="invalid-feedback">Please enter customer name</div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Phone <span class="text-danger">*</span></label>
                            <input type="tel" class="form-control" name="phone" required pattern="[0-9]{10,}">
                            <div class="invalid-feedback">Please enter a valid phone number</div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="email" required>
                            <div class="invalid-feedback">Please enter a valid email address</div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Address <span class="text-danger">*</span></label>
                            <textarea class="form-control" name="address" rows="2" required></textarea>
                            <div class="invalid-feedback">Please enter customer address</div>
                        </div>
                        <?php if ($creditSystemEnabled): ?>
                        <div class="mb-3">
                            <label class="form-label">Credit Limit <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><?php echo $currency; ?></span>
                                <input type="number" class="form-control" name="credit_limit" 
                                       value="<?php echo $defaultCreditLimit; ?>" 
                                       min="0" max="<?php echo $maxCreditLimit; ?>" step="0.01" required>
                            </div>
                            <div class="invalid-feedback">Please enter a valid credit limit</div>
                            <small class="text-muted">Maximum credit limit is <?php echo $currency . ' ' . number_format($maxCreditLimit, 2); ?></small>
                        </div>
                        <?php endif; ?>
                        <div class="text-end">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_customer" class="btn btn-primary" data-permission="add_customer" data-permission-label="Add Customer">Add Customer</button>
                    </div>
                </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-success text-white border-0">
                    <h5 class="modal-title d-flex align-items-center" id="successModalLabel">
                        <svg class="me-2" width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                            <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                        </svg>
                        Success
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center py-4" id="successModalBody">
                    <!-- Success message will be injected here -->
                </div>
                <div class="modal-footer border-0 justify-content-center">
                    <div class="progress w-100" style="height: 3px;">
                        <div class="progress-bar bg-success" id="successProgress" role="progressbar" style="width: 100%"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    // Define currency variable from PHP
    const currency = '<?php echo $currency; ?>';

    document.addEventListener('DOMContentLoaded', function() {
        // Handle Add Customer form validation and submission
        const addCustomerForm = document.getElementById('addCustomerForm');
        if (addCustomerForm) {
            // Add validation function
            function validateAddCustomerForm() {
                const nameField = addCustomerForm.querySelector('input[name="name"]');
                const phoneField = addCustomerForm.querySelector('input[name="phone"]');
                const emailField = addCustomerForm.querySelector('input[name="email"]');
                const addressField = addCustomerForm.querySelector('textarea[name="address"]');
                const creditLimitField = addCustomerForm.querySelector('input[name="credit_limit"]');
                
                let isValid = true;
                let errors = [];
                
                // Validate Name
                if (!nameField.value.trim()) {
                    nameField.classList.add('is-invalid');
                    errors.push('Customer name is required');
                    isValid = false;
                } else {
                    nameField.classList.remove('is-invalid');
                    nameField.classList.add('is-valid');
                }
                
                // Validate Phone
                const phonePattern = /^[0-9]{10,}$/;
                if (!phoneField.value.trim()) {
                    phoneField.classList.add('is-invalid');
                    errors.push('Phone number is required');
                    isValid = false;
                } else if (!phonePattern.test(phoneField.value.trim())) {
                    phoneField.classList.add('is-invalid');
                    errors.push('Please enter a valid phone number (at least 10 digits)');
                    isValid = false;
                } else {
                    phoneField.classList.remove('is-invalid');
                    phoneField.classList.add('is-valid');
                }
                
                // Validate Email
                const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailField.value.trim()) {
                    emailField.classList.add('is-invalid');
                    errors.push('Email address is required');
                    isValid = false;
                } else if (!emailPattern.test(emailField.value.trim())) {
                    emailField.classList.add('is-invalid');
                    errors.push('Please enter a valid email address');
                    isValid = false;
                } else {
                    emailField.classList.remove('is-invalid');
                    emailField.classList.add('is-valid');
                }
                
                // Validate Address
                if (!addressField.value.trim()) {
                    addressField.classList.add('is-invalid');
                    errors.push('Customer address is required');
                    isValid = false;
                } else {
                    addressField.classList.remove('is-invalid');
                    addressField.classList.add('is-valid');
                }
                
                // Validate Credit Limit (if field exists)
                if (creditLimitField) {
                    const creditValue = parseFloat(creditLimitField.value);
                    if (isNaN(creditValue) || creditValue < 0) {
                        creditLimitField.classList.add('is-invalid');
                        errors.push('Please enter a valid credit limit');
                        isValid = false;
                    } else {
                        creditLimitField.classList.remove('is-invalid');
                        creditLimitField.classList.add('is-valid');
                    }
                }
                
                addCustomerForm.classList.add('was-validated');
                
                if (!isValid) {
                    // Show error message
                    alert('Please correct the following errors:\n\n• ' + errors.join('\n• '));
                    
                    // Focus on first invalid field
                    const firstInvalidField = addCustomerForm.querySelector('.is-invalid');
                    if (firstInvalidField) {
                        firstInvalidField.focus();
                    }
                }
                
                return isValid;
            }
            
            // Handle form submission
            addCustomerForm.addEventListener('submit', function(e) {
                // Run validation
                if (!validateAddCustomerForm()) {
                    e.preventDefault();
                    e.stopPropagation();
                    return false;
                }
                
                // If validation passes, allow normal form submission
                return true;
            });
            
            // Real-time validation for better UX
            const fields = addCustomerForm.querySelectorAll('input, textarea');
            fields.forEach(field => {
                field.addEventListener('blur', function() {
                    // Only validate if form has been submitted once
                    if (addCustomerForm.classList.contains('was-validated')) {
                        validateField(this);
                    }
                });
                
                field.addEventListener('input', function() {
                    // Remove invalid state as user types
                    if (this.classList.contains('is-invalid')) {
                        this.classList.remove('is-invalid');
                    }
                });
            });
        }
        
        // Function to validate individual field
        function validateField(field) {
            const fieldName = field.getAttribute('name');
            let isValid = true;
            
            switch(fieldName) {
                case 'name':
                    isValid = field.value.trim() !== '';
                    break;
                case 'phone':
                    const phonePattern = /^[0-9]{10,}$/;
                    isValid = field.value.trim() !== '' && phonePattern.test(field.value.trim());
                    break;
                case 'email':
                    const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    isValid = field.value.trim() !== '' && emailPattern.test(field.value.trim());
                    break;
                case 'address':
                    isValid = field.value.trim() !== '';
                    break;
                case 'credit_limit':
                    const creditValue = parseFloat(field.value);
                    isValid = !isNaN(creditValue) && creditValue >= 0;
                    break;
            }
            
            if (isValid) {
                field.classList.remove('is-invalid');
                field.classList.add('is-valid');
            } else {
                field.classList.remove('is-valid');
                field.classList.add('is-invalid');
            }
            
            return isValid;
        }
        
        // Handle edit customer button clicks
        document.querySelectorAll('.edit-customer').forEach(button => {
            button.addEventListener('click', function() {
                const data = this.dataset;
                const setValue = (id, value) => {
                    const el = document.getElementById(id);
                    if (el) el.value = value;
                };
                setValue('edit_id', data.id);
                setValue('edit_name', data.name);
                setValue('edit_phone', data.phone);
                setValue('edit_email', data.email);
                setValue('edit_address', data.address);
                setValue('edit_credit_limit', data.credit);
                new bootstrap.Modal(document.getElementById('editCustomerModal')).show();
            });
        });

        // Handle edit form submission
        const editForm = document.getElementById('editCustomerForm');
        if (editForm) {
            editForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(this);
                formData.append('edit_customer', '1'); // Add the edit_customer flag
                
                fetch(window.location.href, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Update the table row with new data
                        const row = document.querySelector(`tr[data-customer-id="${formData.get('id')}"]`);
                        if (row) {
                            // Update name and address
                            const nameCell = row.querySelector('.fw-bold');
                            if (nameCell) {
                                nameCell.textContent = formData.get('name');
                            }
                            const addressCell = row.querySelector('.text-muted');
                            if (addressCell) {
                                addressCell.textContent = formData.get('address');
                            }
                            
                            // Update phone and email
                            const contactCell = row.querySelector('td:nth-child(2)');
                            if (contactCell) {
                                contactCell.innerHTML = `
                                    <div>${formData.get('phone')}</div>
                                    <small class="text-muted">${formData.get('email')}</small>
                                `;
                            }
                            
                            // Update credit status if available
                            if (data.data) {
                                const statusCell = row.querySelector('.credit-status');
                                if (statusCell) {
                                    statusCell.innerHTML = `
                                        <div class="credit-status-badge ${data.data.credit_status}">
                                            ${data.data.status_text}
                                        </div>
                                    `;
                                }
                                
                                const creditCell = row.querySelector('.available-credit');
                                if (creditCell) {
                                    creditCell.innerHTML = `
                                        <div class="${data.data.available_credit < 0 ? 'text-danger' : 'text-success'}">
                                            ${currency} ${Math.abs(data.data.available_credit).toFixed(2)}
                                        </div>
                                    `;
                                }
                            }
                        }
                        
                        // Close edit modal and show success modal
                        const editModal = bootstrap.Modal.getInstance(document.getElementById('editCustomerModal'));
                        editModal.hide();
                        
                        showSuccessModal(data.message);

                    } else {
                        alert(data.message || 'An error occurred while updating the customer.');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while updating the customer.');
                });
            });
        }
    });

    // Function to show success modal with auto-dismiss
    function showSuccessModal(message) {
        const successModal = new bootstrap.Modal(document.getElementById('successModal'));
        document.getElementById('successModalBody').textContent = message;
        
        // Reset progress bar
        const progressBar = document.getElementById('successProgress');
        progressBar.style.width = '100%';
        
        successModal.show();
        
        // Start progress bar animation
        setTimeout(() => {
            progressBar.style.width = '0%';
        }, 100);
        
        // Auto-dismiss after 2 seconds
        setTimeout(() => {
            successModal.hide();
        }, 2000);
    }

    // Display success/error messages
    <?php if ($success): ?>
        showSuccessModal('<?php echo addslashes($success); ?>');
    <?php endif; ?>
    <?php if ($error): ?>
        alert('<?php echo addslashes($error); ?>');
    <?php endif; ?>

    // Auto-filter functionality
    const filterForm = document.getElementById('filterForm');
    const searchInput = document.getElementById('searchInput');
    const categorySelect = document.getElementById('categorySelect');
    const sortSelect = document.getElementById('sortSelect');
    const orderSelect = document.getElementById('orderSelect');
    
    let searchTimeout;
    
    // Function to submit the filter form
    function submitFilters() {
        filterForm.submit();
    }
    
    // Add event listeners for immediate filtering
    if (categorySelect) {
        categorySelect.addEventListener('change', submitFilters);
    }
    
    if (sortSelect) {
        sortSelect.addEventListener('change', submitFilters);
    }
    
    if (orderSelect) {
        orderSelect.addEventListener('change', submitFilters);
    }
    
    // Add debounced search input
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(submitFilters, 500); // 500ms delay for search
        });
        
        // Submit immediately on Enter key
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                clearTimeout(searchTimeout);
                submitFilters();
            }
        });
    }
    </script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html> 